/*
 * ExNihilo 3D Engine
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Please read AUTHORS file !!!
 * 
 * $Id: ExCImage.cpp,v 1.2 2002/08/01 19:37:50 binny Exp $
 *
 */

#include "ExCImage.h"

bool LoadBMP(tImage *image,std::string strFileName)
{
	return LoadBMP(image,strFileName.data());
}

bool LoadTGA(tImage *image,std::string strFileName)
{
	return LoadTGA(image,strFileName.data());
}

bool LoadJPG(tImage *image,std::string strFileName)
{
	return LoadJPG(image,strFileName.data());
}

bool LoadBMP(tImage *image,const char *strFileName)
{
	//---------------------------------------------
	FILE *file;
	unsigned long size;                 // size of the image in uchars.
	unsigned long i;                    // standard counter.
	unsigned short int planes;          // number of planes in image (must be 1) 
	unsigned short int channels;        // number of bits per pixel (must be 24)
	char temp;                          // temporary color storage for bgr-rgb conversion.

	// make sure the file is there.
	if ((file = fopen(strFileName, "rb"))==NULL)
	{
		printf("File Not Found : %s\n",strFileName);
		return false;
	}
#ifdef UNIX_SRC
	unsigned short int is_bmp;
	if (! fread (&is_bmp, sizeof (short int), 1, file))
	{
		printf("cannot read %s.\n", strFileName);
		return false;
	}
	
	/* check if file is a bitmap */
	if (is_bmp != 19778) 
	{
		printf("%s is not a valid bmp file.\n", strFileName);
		return false;
	}
	
	fseek (file, 8, SEEK_CUR);
	/* get the position of the actual bitmap data */
	long int bfOffBits;
	if (! fread (&bfOffBits, sizeof (long int), 1, file)) 
	{
		printf("Error reading %s.\n", strFileName);
		return false;
	}
	/*sprintf(buffer,"Data at offset: %ld.", bfOffBits);
	WriteToConsol(buffer);*/
	
	// seek through the bmp header, up to the width/height:
	fseek(file, 4, SEEK_CUR);
#else
	fseek(file, 18, SEEK_SET);
#endif
	// read the width
	if ((i = fread(&image->sizeX, 4, 1, file)) != 1) 
	{
		printf("Error reading width from %s.\n", strFileName);
		return false;
	}
			
	// read the height 
	if ((i = fread(&image->sizeY, 4, 1, file)) != 1) 
	{
		printf("Error reading height from %s.\n", strFileName);
		return false;
	}
	
	
	// calculate the size (assuming 24 bits or 3 uchars per pixel).
	size = image->sizeX * image->sizeY * 3;

	// read the planes
	if ((fread(&planes, 2, 1, file)) != 1)
	{
		printf("Error reading planes from %s.\n", strFileName);
		return false;
	}

	if (planes != 1) 
	{
		printf("Planes from %s is not 1: %u.\n", strFileName, planes);
		return false;
	}

	// read the channels
	if ((i = fread(&channels, 2, 1, file)) != 1) 
	{
		printf("Error reading channels from %s.\n", strFileName);
		return false;
	}
	if (channels != 24) 
	{
		printf("channels from %s is not 24: %u\n", strFileName, channels);
		return false;
	}
	
	// seek past the rest of the bitmap header.
	fseek(file, 24, SEEK_CUR);
	// read the data. 
	image->data = new unsigned char[size];
	//char *foo = new char[size];
	//image->data = foo;
	
	if (image->data == NULL) 
	{
		printf("Error allocating memory for color-corrected image data");
		return false;	
	}

	if ((i = fread(image->data, size, 1, file)) != 1)
	{
		printf("Error reading image data from %s.\n", strFileName);
		return false;
	}

	for (i=0;i<size;i+=3) 
	{ 
		// reverse all of the colors. (bgr -> rgb)
		temp = image->data[i];
		image->data[i] = image->data[i+2];
		image->data[i+2] = temp;
	}
	//---------------------------------------------
	image->type=GL_RGB;
	return true;
}

bool LoadTGA(tImage *image,const char *strFileName)
{
	WORD width = 0, height = 0;			// The dimensions of the image
	GLbyte length = 0;					// The length in bytes to the pixels
	GLbyte imageType = 0;					// The image type (RLE, RGB, Alpha...)
	GLbyte bits = 0;						// The bits per pixel for the image (16, 24, 32)
	FILE *pFile = NULL;					// The file pointer
	int channels = 0;					// The channels of the image (3 = RGA : 4 = RGBA)
	int stride = 0;						// The stride (channels * width)
	int i = 0;							// A counter

	// Open a file pointer to the targa file and check if it was found and opened 
	if((pFile = fopen(strFileName, "rb")) == NULL) 
	{
		// Display an error message saying the file was not found, then return false
		printf("Unable to load TGA File!");
		return false;
	}
		
	// Allocate the structure that will hold our eventual image data (must free it!)
	image = (tImage*)malloc(sizeof(tImage));

	// Read in the length in bytes from the header to the pixel data
	fread(&length, sizeof(GLbyte), 1, pFile);
	
	// Jump over one byte
	fseek(pFile,1,SEEK_CUR); 

	// Read in the imageType (RLE, RGB, etc...)
	fread(&imageType, sizeof(GLbyte), 1, pFile);
	
	// Skip past general information we don't care about
	fseek(pFile, 9, SEEK_CUR); 

	// Read the width, height and bits per pixel (16, 24 or 32)
	fread(&width,  sizeof(WORD), 1, pFile);
	fread(&height, sizeof(WORD), 1, pFile);
	fread(&bits,   sizeof(GLbyte), 1, pFile);
	
	// Now we move the file pointer to the pixel data
	fseek(pFile, length + 1, SEEK_CUR); 

	// Check if the image is RLE compressed or not
	if(imageType != TGA_RLE)
	{
		// Check if the image is a 24 or 32-bit image
		if(bits == 24 || bits == 32)
		{
			// Calculate the channels (3 or 4) - (use bits >> 3 for more speed).
			// Next, we calculate the stride and allocate enough memory for the pixels.
			channels = bits / 8;
			stride = channels * width;
			image->data = ((unsigned char*)malloc(sizeof(unsigned char)*stride*height));

			// Load in all the pixel data line by line
			for(uint y = 0; y < height; y++)
			{
				// Store a pointer to the current line of pixels
				unsigned char *pLine = &(image->data[stride * y]);

				// Read in the current line of pixels
				fread(pLine, stride, 1, pFile);
			
				// Go through all of the pixels and swap the B and R values since TGA
				// files are stored as BGR instead of RGB (or use GL_BGR_EXT verses GL_RGB)
				for(i = 0; i < stride; i += channels)
				{
					int temp     = pLine[i];
					pLine[i]     = pLine[i + 2];
					pLine[i + 2] = temp;
				}
			}
		}
		// Check if the image is a 16 bit image (RGB stored in 1 unsigned short)
		else if(bits == 16)
		{
			unsigned short pixels = 0;
			int r=0, g=0, b=0;

			// Since we convert 16-bit images to 24 bit, we hardcode the channels to 3.
			// We then calculate the stride and allocate memory for the pixels.
			channels = 3;
			stride = channels * width;
			image->data = ((unsigned char*)malloc(sizeof(unsigned char)*stride*height));

			// Load in all the pixel data pixel by pixel
			for(uint i = 0; i < width*height; i++)
			{
				// Read in the current pixel
				fread(&pixels, sizeof(unsigned short), 1, pFile);
				
				// Convert the 16-bit pixel into an RGB
				b = (pixels & 0x1f) << 3;
				g = ((pixels >> 5) & 0x1f) << 3;
				r = ((pixels >> 10) & 0x1f) << 3;
				
				// This essentially assigns the color to our array and swaps the
				// B and R values at the same time.
				image->data[i * 3 + 0] = r;
				image->data[i * 3 + 1] = g;
				image->data[i * 3 + 2] = b;
			}
		}	
		// Else return a false for a bad or unsupported pixel format
		else
			return false;
	}
	// Else, it must be Run-Length Encoded (RLE)
	else
	{
		// Create some variables to hold the rleID, current colors read, channels, & stride.
		GLbyte rleID = 0;
		int colorsRead = 0;
		channels = bits / 8;
		stride = channels * width;

		// Next we want to allocate the memory for the pixels and create an array,
		// depending on the channel count, to read in for each pixel.
		image->data = ((unsigned char*)malloc(sizeof(unsigned char)*stride*height));
		GLbyte *pColors = ((GLbyte*)malloc(sizeof(GLbyte)*channels));

		// Load in all the pixel data
		while((unsigned)i < width*height)
		{
			// Read in the current color count + 1
			fread(&rleID, sizeof(GLbyte), 1, pFile);
			
			// Check if we don't have an encoded string of colors
			if(rleID < (unsigned)128)
			{
				// Increase the count by 1
				rleID++;

				// Go through and read all the unique colors found
				while(rleID)
				{
					// Read in the current color
					fread(pColors, sizeof(GLbyte) * channels, 1, pFile);

					// Store the current pixel in our image array
					image->data[colorsRead + 0] = pColors[2];
					image->data[colorsRead + 1] = pColors[1];
					image->data[colorsRead + 2] = pColors[0];

					// If we have a 4 channel 32-bit image, assign one more for the alpha
					if(bits == 32)
						image->data[colorsRead + 3] = pColors[3];

					// Increase the current pixels read, decrease the amount
					// of pixels left, and increase the starting index for the next pixel.
					i++;
					rleID--;
					colorsRead += channels;
				}
			}
			// Else, let's read in a string of the same character
			else
			{
				// Minus the 128 ID + 1 (127) to get the color count that needs to be read
				rleID -= 127;

				// Read in the current color, which is the same for a while
				fread(pColors, sizeof(GLbyte) * channels, 1, pFile);

				// Go and read as many pixels as are the same
				while(rleID)
				{
					// Assign the current pixel to the current index in our pixel array
					image->data[colorsRead + 0] = pColors[2];
					image->data[colorsRead + 1] = pColors[1];
					image->data[colorsRead + 2] = pColors[0];

					// If we have a 4 channel 32-bit image, assign one more for the alpha
					if(bits == 32)
						image->data[colorsRead + 3] = pColors[3];

					// Increase the current pixels read, decrease the amount
					// of pixels left, and increase the starting index for the next pixel.
					i++;
					rleID--;
					colorsRead += channels;
				}
				
			}
				
		}
	}

	// Close the file pointer that opened the file
	fclose(pFile);

	// Fill in our tImage structure to pass back
	image->channels = channels;
	image->sizeX    = width;
	image->sizeY    = height;

	// Return the TGA data (remember, you must free this data after you are done)
	return true;
}

bool LoadJPG(tImage *image,const char *strFileName)
{
	return true;
}

/*DecodeJPG(jpeg_decompress_struct* cinfo, tImage *image)
{
}*/
